package dataFit;

import visad.*;                                               // The actual interpolation algorithm
import dataFit.InitInterpolation;          			  // To initialise the interpolation
import dataFit.NotSameLengthException;  				  // Interpolation exception
import java.rmi.RemoteException;                              // This is used by VisAD

public class Interpolation2DLinear
{
  // Declare variables
  private RealType xcoord, ycoord, zcoord;
  private RealTupleType domain_tuple;

  // The function (domain_tuple -> zcoord )
  private FunctionType func_domain_zcoord;

  // The Data values for the domain are represented by the Set
  private Set domain_set, intp_set;

  // The Data class FlatField
  private FlatField vals_ff;

  // The DataReference from data to display
  private DataReferenceImpl data_ref;

  // The value used for scaling data
  float Min_Xf, Max_Xf, Min_Yf, Max_Yf;

  /**
   * Interpolation2DLinear(double[] Xf, double[] Yf, double[] Zf) is the
   * constructor of the class Interpolation2DLinear. It creates and instanciates
   * some particular variables required by VisAD to initialise the function z=F(x,y)
   * and the domain where its value is known. It also initialises numerically the
   * 2D-function to be interpolated. The vectors Xf and Yf define the definition
   * domain of the function F, and the vector Zf contains the values taken by F.
   */
  public Interpolation2DLinear(double[] Xf, double[] Yf, double[] Zf)
  throws RemoteException, VisADException, NotSameLengthException
  {
    // Create the coordinates (x,y,z) system
    // Use VisAD RealType(String name);
    xcoord = InitInterpolation.xcoord;
    ycoord = InitInterpolation.ycoord;
    zcoord = InitInterpolation.zcoord;
    domain_tuple = new RealTupleType(xcoord , ycoord);

    // Create a FunctionType (domain_tuple -> zcoord)
    // Use FunctionType(MathType domain, MathType range);
    func_domain_zcoord = new FunctionType(domain_tuple, zcoord);

    // Check that the length of all 3 arrays is the same
    int the_length1 = Xf.length;
    int the_length2 = Yf.length;
    int the_length3 = Zf.length;
    if ((the_length1 != the_length2)||(the_length1 != the_length3))
    {
      throw new NotSameLengthException("[dataFit][Interpolation2DLinear][constructor]: Arrays must have the same length.");
    }

    // Find out the Min and Max of Xf and Yf for
    // Converting data to the right scale later
    Min_Xf=(float)Xf[0];
    Min_Yf=(float)Yf[0];
    Max_Xf=(float)Xf[0];
    Max_Yf=(float)Yf[0];
    for (int i=0; i<the_length1; i++)
    {
      if (Xf[i]<=Min_Xf)
      {
        Min_Xf=(float)Xf[i];
      }
      if (Yf[i]<=Min_Yf)
      {
        Min_Yf=(float)Yf[i];
      }
      if (Xf[i]>=Max_Xf)
      {
        Max_Xf=(float)Xf[i];
      }
      if (Yf[i]>=Max_Yf)
      {
        Max_Yf=(float)Yf[i];
      }
    }

    // Create the domain Set
    // Conversion double-->float and scale the value
    float[][] my_sample = new float[2][the_length1];
    for(int i=0; i < the_length1; i++)
    {
      my_sample[0][i]=((float)Xf[i]-Min_Xf)/(Max_Xf-Min_Xf);
      my_sample[1][i]=((float)Yf[i]-Min_Yf)/(Max_Yf-Min_Yf);
    }
    domain_set = new Irregular2DSet(domain_tuple, my_sample);

    // We create another array that will contains the values of zcoord
    float[][] flat_samples = new float[1][the_length1];
    for(int i=0; i < the_length1; i++)
    {
      flat_samples[0][i]=(float)Zf[i];
    }

    // Create a FlatField
    // Use FlatField(FunctionType type, Set domain_set)
    vals_ff = new FlatField( func_domain_zcoord, domain_set);

    // ...and put the values above into it
    // Note the argument false, meaning that the array won't be copied
    vals_ff.setSamples( flat_samples , false );

    // Inform the users that the class has been instanciated
    //System.out.println("Object of type Interpolation2DLinear created...");
  }

  /** The method compute(double[] Xd, double[] Yd) computes the
   * the interpolation by using a weighted average algorithm. When the
   * computing starts, the object on which the method is applied automatically
   * calls a Delaunay algorithm to generate the mesh that will support
   * the interpolation itself. (Xd,Yd) are the arrays of points where the function is
   * interpolated. This method is a wrapper of the VisAD method resample() and allows
   * interpolation at multiple points at a time. The compute method takes into account
   * the scalling factor proper.
   */
  public double[] compute(double[] Xd, double[] Yd)
  throws RemoteException, VisADException, NotSameLengthException
  {
    // Checking that both arrays have same length
    int the_length4 = Xd.length;
    int the_length5 = Yd.length;
    if (the_length4 != the_length5)
    {
      throw new NotSameLengthException("[dataFit][Interpolation2DLinear][compute]: Arrays must have the same length.");
    }

    // Conversion double-->float
    float[][] intp_sample = new float[2][the_length4];
    for(int i=0; i < the_length4; i++)
    {
      intp_sample[0][i]=((float)Xd[i]-Min_Xf)/(Max_Xf-Min_Xf);
      intp_sample[1][i]=((float)Yd[i]-Min_Yf)/(Max_Yf-Min_Yf);
    }
    intp_set = new Irregular2DSet(domain_tuple, intp_sample);

    // Compute the actual interpolation
    float[][] result = new float[1][the_length4];
    result=(vals_ff.resample(intp_set,DataImpl.WEIGHTED_AVERAGE,DataImpl.INDEPENDENT)).getFloats();

    // Return the results
    double[] final_result = new double[the_length4];
    for(int i=0; i<the_length4; i++)
    {
      final_result[i] = (double)result[0][i];
    }
    return final_result;
  }

  /**
   * The method evaluate(double Xd, double Yd) computes the
   * the interpolation by using a weighted average algorithm. The data are
   * first scaled by (Xi-Min_X)/(Max_X-Min_X) (and the same for the array Y). When the
   * computing starts, the object on which the method is applied automatically
   * calls a Delaunay algorithm to generate the mesh that will support
   * the interpolation itself. (Xd,Yd) is the point where the function is
   * interpolated. This method is a wrapper of the VisAD method evaluate() and
   * only allows interpolation at a single point at a time.
   */
  public double evaluate (double Xd, double Yd)
  throws RemoteException, VisADException, NotSameLengthException
  {
    Real xintp = new Real(xcoord, (Xd-(double)Min_Xf)/((double)Max_Xf-(double)Min_Xf));
    Real yintp = new Real(ycoord, (Yd-(double)Min_Yf)/((double)Max_Yf-(double)Min_Yf));
    Real[] valint = new Real[] {xintp, yintp};
    RealTuple intp_tuple = new RealTuple(valint);
    Data result_temp = vals_ff.evaluate(intp_tuple,DataImpl.WEIGHTED_AVERAGE,DataImpl.INDEPENDENT);
    Real temp = (Real) result_temp;
    double result = temp.getValue();
    return result;
  }

 
  /**
   * Main method. Only use for a demo of the Interpolation2DLinear class.
   */
  public static void main(String[] args)
  throws VisADException, RemoteException, NotSameLengthException
  {
      // Example of creation of an object of type Interpolation2DLinear
      double[] my_x = new double[] {-2, -1, 0, 1, 3, 0};
      double[] my_y = new double[] {0.5, 0, 1, 0, 1, -3};
      double[] my_z = new double[] {0.62, 0.73, 0.58, 0.79, 0.34, 0.11};

      Interpolation2DLinear myInterpolation = new Interpolation2DLinear(my_x,my_y,my_z);

      // Example of interpolation with resample
      double[] my_xd = new double[] {0., -0.1};
      double[] my_yd = new double[] {0., 0.5};
      double[] result = new double[2];
      System.out.println("Now I interpolate at (0,0) and (-0.1,0.5) by using the method compute");
      result = myInterpolation.compute(my_xd, my_yd);
      System.out.println(result[0]);
      System.out.println(result[1]);

      // Example of interpolation with evaluate
      System.out.println("interpolation at (0,0) with evaluate: "+myInterpolation.evaluate(0.,0.));
  }
}